package com.ups.xmlsdk.util.config;

/*
**
** The use, disclosure, reproduction, modification, transfer, or transmittal
** of this work for any purpose in any form or by any means without the
** written permission of United Parcel Service is strictly prohibited.
**
** Confidential, Unpublished Property of United Parcel Service.
** Use and Distribution Limited Solely to Authorized Personnel.
**
** Copyright 1997,1998 United Parcel Service of America, Inc.  All Rights Reserved.
**
**
** File        : AppConfig.java
**
**
** Package     : com.ups.xml.util.config
**
** Classes     : AppConfig
*
** Description : Application-wideinformation class.
**		 Provide  info or options read from a xml fomatted config file.
**       If a config file is not set, uses the xml default in
**       /export/config/xml/sdk.xml
**
**
** Usage       :
**
** Notes       :
**
*/
import java.util.*;
import java.io.*;
import com.ups.xmlsdk.util.logging.*;
import com.ups.xmlsdk.util.CommonFuncs;
import org.w3c.dom.Node;
import org.w3c.dom.Document;
import com.ibm.xml.parser.Parser;

/*
 * This AppConfig will use an XML configuration file to load its configuration options
 *
 */

public class AppConfig implements LogDefs
{
	//	protected static final String defaultConfigFile = File.separator + "export" + File.separator + "config" + File.separator + "xml" + File.separator + "sdkConfig.xml";
	protected static final String defaultConfigFile = "d:" + File.separator + "xmlsdk" + File.separator + "conf" + File.separator + "sdkConfig.xml";
	protected Hashtable optionsTable = null;
	protected static final String SYNCSTR = "";
	protected static AppConfig ac = null;
	protected static String configFileName = defaultConfigFile;
	protected ConfigUpdater cu = null;
	protected int configUpdateInterval = 300000;
	protected Logger logger = DefaultLogFactory.instance().getDefaultLogger();
	protected static final String MESSAGE_BASE = "com.ups.common.resources.strings.";
	protected static final String CREDIT_CARD_MESSAGES_BASE = MESSAGE_BASE + "creditcardmessages.CreditCardMessages";
	protected static final String VALIDATOR_MESSAGES_BASE = MESSAGE_BASE + "validatormessages.ValidatorMessages";
	/**
 	* Constructor. An application should only create one of these. However,
 	* we don't bother trying to enforce this; since callers can't call it.
 	*/
	protected AppConfig() throws Exception
	{
		optionsTable  = new Hashtable();
		init();
	}
	/**
	 * Function to return the specified boolean value
	 * @return boolean value for the specified key
	 */
	public boolean getBoolean(String key) throws Exception,
		ConfigValueNotFoundException
	{
		boolean bReturn = false;

		String tmpValue = getValueFromOptions(key);
		if (tmpValue.equalsIgnoreCase("yes") || tmpValue.equalsIgnoreCase("true"))
		{
	   		bReturn = true;
		}

		return bReturn;
	}
	public static String getConfigFileName()
	{
		return configFileName;
	}
   /**
	* Gets the credit card messages resource bundle base name.
	*/
   static public String getCreditCardMessagesBase()
   {
	   return (CREDIT_CARD_MESSAGES_BASE);
   }                  
	/**
	 * Function to return the specified double value
	 * @return double value for the specified key
	 */
	public double getDouble(String key) throws Exception,
		ConfigValueNotFoundException
	{
		String strValue = getValueFromOptions(key);
		double lReturn =  new Double(strValue).doubleValue();
		return lReturn;
	}
	/**
	 * Function to return the specifed int value
	 * @return int value for the specified key
	 */
	public int getInt(String key) throws Exception,
		ConfigValueNotFoundException
	{
		String strValue = getValueFromOptions(key);
		int retVal = new Integer(strValue).intValue();
		return retVal;
	}
	/**
	 * Function to return the specified long value
	 * @return long value for the specified key
	 */
	public long getLong(String key) throws Exception,
		ConfigValueNotFoundException
	{
		String strValue = getValueFromOptions(key);
		long lReturn =  new Long(strValue).longValue();
		return lReturn;
	}
	/**
	 * Function to return the specified String value
	 * @return String value for the specified key
	 */
	public String getString(String key) throws Exception,
		ConfigValueNotFoundException
	{
		String retVal = getValueFromOptions(key);
		return retVal;
	}
   /**
	* Gets the validator messages resource bundle base name.
	*/
   static public String getValidatorMessagesBase()
   {
	   return (VALIDATOR_MESSAGES_BASE);
   }                  
	//-------------------------------------------------------
	// Reads value from app options file
	//-------------------------------------------------------

	protected String getValueFromOptions(String theKey) throws Exception,
		ConfigValueNotFoundException
	{

		String theValue = (String) optionsTable.get(theKey);
		if ( theValue == null )
		{
			throw (new ConfigValueNotFoundException(theKey));
		}

		return theValue;
	}
	public void init() throws Exception
	{
	  logger = DefaultLogFactory.instance().getDefaultLogger();

		loadOptions();
		try
		{
			configUpdateInterval = getInt("ConfigUpdateInterval");
			if (configUpdateInterval <= 0)
			{
				configUpdateInterval = 300000;
			}
		}
		catch (Exception e)
		{
			// don't log this, write to stdout since we could be recursion
	   		System.out.println("Exception in AppConfig.init(): " +
	   			CommonFuncs.getStackTraceString(e));
		}

		// Start the ConfigUpdater
		restart();
	}
	/**
 	* Lazily instantiate a validator and return it.
 	*/
	public static AppConfig Instance() throws Exception
	{
		if (ac == null)
		{
		   synchronized (SYNCSTR)
		   {
			   if (ac == null)
			   {
				   ac = new AppConfig();
			   }
		   }
		}

		return ac;
	}
protected void loadOptions() throws Exception
{
	try
	{
		// get file
		FileInputStream fis = new FileInputStream(configFileName);

		// com.ibm.xml.parser.Parser
		System.out.println("Begin Parse");
		Parser parser = new Parser(configFileName);
		parser.parse(new org.xml.sax.InputSource(fis));
		if (parser.getNumberOfErrors() > 0)
		{
			logger.log(parser.getNumberOfErrors() + " Errors Found in SDKConfig.xml");
		}

		//	load hash from the xml stream
		Document doc = parser.getDocument();
		Node root = doc.getFirstChild();
		Node child = null;
		for (child = root.getFirstChild(); child != null; child = child.getNextSibling())
		{
			try
			{
				if (child.getNodeValue() != null)
				{
					optionsTable.put(child.getNodeName(), child.getNodeValue());
				}
				Node pup = null;
				for (pup = child.getFirstChild(); pup != null; pup = pup.getNextSibling())
				{
					try
					{
						if (pup.getNodeValue() != null)
						{
							optionsTable.put(pup.getNodeName(), pup.getNodeValue());
						}
						Node sub = null;
						for (sub = pup.getFirstChild(); sub != null; sub = sub.getNextSibling())
						{
							try
							{
								if (sub.getNodeValue() != null)
								{
									optionsTable.put(sub.getNodeName(), sub.getNodeValue());
								}
							} catch (Exception e4)
							{
							}
						}
					} catch (Exception e3)
					{
					}
				}
			} catch (Exception e2)
			{
			}
		}
		fis.close(); // close the InputStream
	} catch (Exception e)
	{
		logger.error("Exception occured in loadOptions(): " + e.getMessage());
	}
}
	// Notify registered clients that AppConfig
	protected void notifyChanged()
	{
		System.out.println("Object changed");
	}
   /**
	* Restarts the AppConfig - allows external start and stop of the ConfigUpdater thread.
	*/
   public void restart()
   {
	   terminate();

	   cu = new ConfigUpdater(this, configUpdateInterval);
	   cu.start();
   }                  
	/**
	 * Function to set the specified boolean value
	 */
	public void setBoolean(String key, boolean val)
	{
		String value = new Boolean(val).toString();
		optionsTable.put(key, value);
	}
	// This method needs to be called before the Instance method is used.
	public static void setConfigFileName(String cfName)
	{
		configFileName = cfName;
		if (cfName != null)
		{
			try
			{
				if (ac != null)
				{
					ac.loadOptions();
				}
			}
			catch (Exception e)
			{
				System.out.println("AppConfig, setConfigFileName error: " +
					e.getMessage());
			}
		}
	}
	/**
	 * Function to set the specifed double value
	 */
	public void setDouble(String key, double val)
	{
		String value = new Double(val).toString();
		optionsTable.put(key, value);
	}
	/**
	 * Function to set the specified int value
	 */
	public void setInt(String key, int val)
	{
		String value = new Integer(val).toString();
		optionsTable.put(key, value);
	}
	/**
	 * Function to set the specified long value
	 */
	public void setLong(String key, long val)
	{
		String value = new Long(val).toString();
		optionsTable.put(key, value);
	}
	/**
	 * Function to set the specified String value
	 */
	public void setString(String key, String val)
	{
		optionsTable.put(key, val);
	}
   /**
	* Terminates the AppConfig - allows external start and stop of the ConfigUpdater thread.
	*/
   public void terminate()
   {
	   if (cu != null && cu.isAlive())
	   {
		   cu.setStop();
	   }
   }                  
}
