package com.ups.xmlsdk.util;

import java.io.*;
import java.util.*;
import java.net.*;
import com.ibm.samplepackage.*;
import com.ibm.samplepackage.https.*;
import com.ups.xmlsdk.util.logging.DefaultLogFactory;
import com.ups.xmlsdk.util.logging.Logger;

/**
 * The XmlTransmitter will transmit an HTTP/HTTPS post with the StringBuffer provided as 
 * the data of the post message.  The XmlTransmitter must be constructed with a URL or
 * IP address and a protocol to use for transmitting the message.
 */
public class XmlTransmitter
{
	private String hostname;
	private String protocol;
	private String prefix;
	private String proxyHost;
	private String proxyPort;
	private String keyLocation;
	private StringBuffer XmlIn;
	private StringBuffer XmlOut;
	private static java.lang.String username;
	private static java.lang.String password;
	private static java.lang.String encodedPass;

	// Logging
	private Logger logger;
	private final static java.lang.String __CLASS__ = "XMLTransmitter::";
	private final static java.lang.String METHOD_ContactService = "contactService(String, String)";
	private final static java.lang.String METHOD_readURLConnection = "readURLConnection(URLConnection)";
/**
 * Constructs a new XmlTransmitter for purposes of HTTP posts not inhibited by
 * a Proxy Server or FireWall.
 * @param hostname java.lang.String
 * @param protocol java.lang.String
 * @param keyring java.lang.String
 * 
 */
public XmlTransmitter(String hostname, String protocol, String keyring)
{
	this.hostname = hostname;
	this.protocol = protocol;
	this.keyLocation = keyring;

	// Logging
	try
	{
		logger = DefaultLogFactory.instance().getOrAddLogger("XMLTransmitter");
	} catch (Exception e)
	{
		System.out.println("XMLTransmitter() error retrieving logger!");
	}
}
/**
 * Constructs a new XmlTransmitter for purposes of HTTP posts and allows the addition of
 * Proxy information for access through a Proxy Server or FireWall.  The software assumes
 * Basic Authentication is used by 99% of Proxy Servers and Firewalls.
 * @param hostname java.lang.String
 * @param protocol java.lang.String
 * @param proxy java.lang.String
 * @param port java.lang.String
 * @param keyring java.lang.String
 * 
 */
public XmlTransmitter(String hostname, String protocol, String keyring, String proxy, String port, String username, String password)
{
	this.hostname = hostname;
	this.protocol = protocol;
	this.keyLocation = keyring;
	if (proxy != null && port != null)
	{
		if (protocol.equalsIgnoreCase("https"))
		{
			System.getProperties().put("https.proxyHost", proxy);
			System.getProperties().put("https.proxyPort", port);
			this.proxyHost = proxy;
			this.proxyPort = port;
			this.username = username;
			this.password = password;
		} else
		{
			System.getProperties().put("proxySet", "true");
			System.getProperties().put("http.proxyHost", proxy);
			System.getProperties().put("http.proxyPort", port);
			java.lang.String pass = username + ":" + password;
			// This handles Basic authentication only (99% of situations are basic)
			this.encodedPass = "Basic " + (new Base64()).base64Encode(pass);
		}
	}

	// Logging
	try
	{
		logger = DefaultLogFactory.instance().getOrAddLogger("XMLTransmitter");
	} catch (Exception e)
	{
		System.out.println("XMLTransmitter() error retrieving logger!");
	}
}
/**
 * This method is used to send the XmlTransmitter information to a designated service.  
 * @param service java.lang.String
 * @param prefix java.lang.String
 * @throws java.lang.Exception
 */
public void contactService(String service, String prefix) throws Exception
{
	logger.debug(__CLASS__ + METHOD_ContactService + "******************************** Started " + service + " " + new Date().toString() + " *************************************");
	try
	{
		// Create new URL and connect
		if (protocol.equalsIgnoreCase("https"))
		{
			setHttpsContext(keyLocation, "sslight");
		}
		URL url = new URL(protocol + "://" + hostname + "/" + prefix + "/" + service);
		HttpURLConnection connection = (HttpURLConnection) url.openConnection();
		logger.debug(__CLASS__ + METHOD_ContactService + "Establishing connection with " + url.toString());

		// Setup HTTP POST parameters
		connection.setDoOutput(true);
		connection.setDoInput(true);
		connection.setUseCaches(false);
		if (this.encodedPass != null)
			connection.setRequestProperty("Proxy-Authorization", this.encodedPass);
		// connection.setFollowRedirects(false);

		// Get POST data from input file
		String queryString = XmlIn.toString();

		// POST data
		OutputStream out = connection.getOutputStream();
		out.write(queryString.getBytes());
		logger.debug(__CLASS__ + METHOD_ContactService + "Transmission sent to " + url.toString() + ":\n" + queryString);
		out.close();

		// get Response Http headers
		try
		{
			int rc = connection.getResponseCode();
			switch (rc)
			{
				case 500 :
					throw new SDKException(SDKException.CONNECTION_ERROR, "Http Response Code of 500 from Online Tools Server");
				case 404 :
					throw new SDKException(SDKException.CONNECTION_ERROR, "Http Response Code of 404 from Online Tools Server");
				case 401 :
					throw new SDKException(SDKException.CONNECTION_ERROR, "Http Response Code of 401 from Online Tools Server");
				case 302 :
					logger.log(__CLASS__ + METHOD_ContactService + "Redirect = " + connection.getHeaderField("Location"));
			}
		} catch (Exception e)
		{
			throw new SDKException(SDKException.CONNECTION_ERROR, "Unknown URL Condition!" + connection.getResponseMessage());
		}
		String data = "";
		// get data from URL connection to return as a StringBuffer

		data = readURLConnection(connection);
		logger.debug("Response = " + data);
		XmlOut = new StringBuffer(data);
	} catch (SDKException sdk)
	{
		logger.error(sdk.toString());
		throw sdk;
	} catch (Exception e1)
	{
		logger.error(e1.toString());
		throw new SDKException(SDKException.CONNECTION_ERROR, "Error during transmission");
	} finally
	{
		logger.debug(__CLASS__ + METHOD_ContactService + "******************************** Finished " + service + " " + new Date().toString() + " *************************************");
	}
}
/**
 * This method returns the xml response, from the XmlTransmitter's URL, to the caller of method.
 * @return java.lang.StringBuffer
 */
public StringBuffer getXml()
{
	return XmlOut;
}
/**
 * This method read all of the data from a URL conection to a String
 */

private static String readURLConnection(URLConnection uc)
{
	StringBuffer buffer = new StringBuffer();
	BufferedReader reader = null;
	try
	{
		reader = new BufferedReader(new InputStreamReader(uc.getInputStream()));
		String line = null;
		int letter = 0;
		while ((letter = reader.read()) != -1)
			buffer.append((char) letter);
	} catch (Exception e)
	{
		System.out.println("Cannot read from URL" + e.toString());
	} finally
	{
		try
		{
			reader.close();
		} catch (IOException io)
		{
			System.out.println("Error closing URLReader!");
		}
	}
	return buffer.toString();
}
/**
 * Insert the method's description here.
 * @param file java.lang.String
 * @param psswd java.lang.String
 */
private static void setHttpsContext(String file, String psswd)
{
	// This commented line enables a context version that is extremely trusty
	//SSLContext context = new com.ibm.samplepackage.MyTrustySSLContext();
	SSLContext context = new com.ibm.samplepackage.SSLContext();
	// context.debug = true;

	SSLPKCS12Token token = new SSLPKCS12Token();
	FileInputStream kin = null;
	try
	{
		byte[] data = new byte[ (int) new File(file).length()];
		(kin = new FileInputStream(file)).read(data);
		token.open(data, psswd);
	} catch (Exception e)
	{
		try
		{
			kin.close();
		} catch (Exception ex)
		{
		}
		System.out.println("https: Cannot load key ring: " + e);
		System.exit(1);
	}
	context.importToken(token);

	// Add the HTTPS protocol handler to system properties
	String p = "com.ibm.samplepackage";
	String s = ((s = System.getProperty("java.protocol.handler.pkgs")) != null ? s + "|" + p : p);
	System.getProperties().put("java.protocol.handler.pkgs", s);
	// Setup the context to be used by HTTPS handler
	com.ibm.samplepackage.https.SSLNetworkClient.setContext(context);
	// HTTPS Proxy is controlled via system properties https.proxyHost/https.proxyPort
	// -- Optionally set these system properties here

	// Resync system properties with HTTPS handler
	// - do this anytime you change system properties related to HTTPS handler
	com.ibm.samplepackage.https.HttpsClient.resetProperties();
	// Setup a callback for proxy authentication (pops up a dialog box)
	com.ibm.samplepackage.https.HttpsURLConnection.setDefaultAuthenticator(new Https(username, password));
	// HTTPS is up and ok now.
}
/**
 * Setter for purposes of giving the XmlTransmitter the request data.
 * @param input java.lang.StringBuffer
 */
public void setXml(StringBuffer input)
{
	this.XmlIn = input;
}
/**
 * toString() returns a String containing the values of the XmlTransmitter's
 * instance variables.
 * @return java.lang.String
 */
public String toString()
{
	StringBuffer infoBuf = new StringBuffer();
	infoBuf.append(protocol + "://");
	infoBuf.append(hostname);
	if (proxyHost != null && proxyPort != null)
	{
		infoBuf.append("    Proxy = " + proxyHost);
		infoBuf.append(":" + proxyPort);
		infoBuf.append(" & SSL KeyRing=");
	}
	if (protocol.equalsIgnoreCase("https"))
	{
		infoBuf.append(keyLocation);
	}
	infoBuf.append("\n");
	infoBuf.append("Input Buffer: " + XmlIn + "\n");
	infoBuf.append("Output Buffer :" + XmlOut + "\n");
	return infoBuf.toString();
}
}
