package com.ups.xmlsdk.util.logging;

/*
**  The use, disclosure, reproduction, modification, transfer, or transmittal
**    of this work for any purpose in any form or by any means without the
**     written permission of United Parcel Service is strictly prohibited.
**
**  Copyright 1999 United Parcel Service of America, Inc.  All Rights Reserved.
*/

import java.io.IOException;

/**
 * General Logging interface.  There is one interface per facility so
 * the user does not have to specify the facility string.
 *
 * The user can also specify a context factory which looks up user context
 * information using the calling thread as a key.  This allows use of the
 * two shorthand log() methods that do not have an explicit context arg.
 * The default factory prints the process and thread IDs.
 */
public class Logger implements LogDefs, LogContextFactory
{
	protected Logger delegate;
	protected String facility;
	protected LogFormatter formatter = new DefaultLogFormatter();
	protected LogContextFactory contextFactory = this;
	boolean open = false;
	protected int level = LOG_ALL;

   	static String[] levels = {
   		"EMERGENCY", "ALERT", "CRITICAL", "ERROR", "WARNING",
   		"NOTICE", "INFO", "DEBUG", "TRACE" };

	protected Logger(String facility)
	{
		this.facility = facility;
	}
	// Create a logger that delegates to another logger
	public Logger(String facility, Logger delegate)
	{
		this(facility);
		if (delegate == null)
		{
			throw new NullPointerException(
				"delegate must be specified for logger");
		}
		this.delegate = delegate;
		this.level = delegate.level;
		this.contextFactory = delegate.contextFactory;
	}
	public final void audit(String msg)
	{
		log(LOG_INFO, msg);
	}
	// Close this instance.  The instance cannot be reused
	public void close() throws IOException
	{
		if (!open) throw new IOException("logger already closed");
		open = false;
	}
	public final void debug(String msg)
	{
		log(LOG_DEBUG, msg);
	}
	// Convenience methods

	public final void error(String msg)
	{
		log(LOG_ERROR, msg);
	}
	// Flush this instance.  Any buffered records are written
	public void flush() throws IOException
	{
	}
	// Return a default user context
	public LogContext getContext()
	{
		return new LogContext();
	}
	// Get the facility for this instance
	public String getFacility()
	{
		return this.facility;
	}
	// Get the record formatter for this instance
	public LogFormatter getFormatter()
	{
		return formatter;
	}
	// Get the priority level threshold on this instance.
	public int getLevel()
	{
		return this.level;
	}
	/*
	 * Default handler for logging exceptions.  We do not want
	 * log methods to throw exceptions since it unnecessarily complicates
	 * the users code.
	 */
	void handleException(LogException e)
	{
		System.out.println("LOGGER EXCEPTION, trace is: ");
		e.printStackTrace();
	}
	// Priority specific log method
	public final void log(int pri, String msg)
	{
		try
		{
		  log(contextFactory.getContext(), facility, pri, msg, CODE_UNSET);
		}
		catch(LogException e)
		{
			handleException(e);
		}
	}
	// Priority and code specific log method
	public final void log(int pri, String msg, long code)
	{
		try
		{
			log(contextFactory.getContext(), facility, pri, msg, code);
		}
		catch(LogException e)
		{
			handleException(e);
		}
	}
	// Context, priority, and facility specific log method, also returns
	// the current logging level
	public int log(LogContext context, String facility, int priority,
		String msg, long code) throws LogException
	{
		if (delegate == null)
		{
			throw new LogException("no delegate for this logger");
		}
		return delegate.log(context, facility, priority, msg, code);
	}
	// Default log method, should be equivalent to DEBUG
	public final void log(String msg)
	{
		try
		{
			log(contextFactory.getContext(), facility, LOG_DEBUG,
					msg, CODE_UNSET);
		}
		catch(LogException e)
		{
			handleException(e);
		}
	}
	/**
	 * Parse a logging level represented as a string.  Return -1 if a
	 * valid level cannot be parsed.
	 */
	public static final int parseLevel(String s)
	{
		String su = s.toUpperCase();
		for (int i=0; i< LOG_ALL; i++)
		{
			if (su.endsWith(levels[i])) return i;
		}
		if (su.endsWith("ALL")) return LOG_ALL;
		return -1;
	}
	// Set the user context factory on this instance, if any.
	public void setContextFactory(LogContextFactory contextFactory)
	{
		this.contextFactory = contextFactory;
	}
	// Set the facility on this instance
	public void setFacility(String facility)
	{
		this.facility = facility;
	}
	// Set the record formatter on this instance
	public void setFormatter(LogFormatter formatter)
	{
		this.formatter = formatter;
	}
	// Set the priority level threshold on this instance.  Messages below
	// the level will not be logged.
	public void setLevel(int level)
	{
		this.level = level;
	}
	public final void trace(String msg)
	{
		log(LOG_TRACE, msg);
	}
}
